// $Id: CLevelMeter.cpp,v 1.6 2007/02/08 21:07:54 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CLevelMeter.hpp"
#include <MathTools/CParameter.hpp>
using Exponent::MathTools::CParameter;
using Exponent::GUI::Controls::CLevelMeter;

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CLevelMeter, CControl);

//	===========================================================================
CLevelMeter::CLevelMeter(IControlRoot *root, const long uniqueId, const CRect &area, const long numberOfSteps) 
		   : CControl(root, uniqueId, area)
		   , m_numberToDraw(0)
		   , m_isVertical(true)
		   , m_numberOfSteps(10)
{
	EXPONENT_CLASS_CONSTRUCTION(CLevelMeter);

	// Are we vertical or horizontal
	m_isVertical = (m_area.getWidth() < m_area.getHeight());

	m_numberOfSteps = numberOfSteps;

	// Check our orientation
	if (m_isVertical)
	{
		// The height of drawing is less than the size of the area
		const long height = m_area.getHeight() - 4;
	
		// There is a two pixel gap
		const long offset = 2 * (numberOfSteps + 1);

		// Total width of each
		const long totalHeight = (height - offset) / numberOfSteps;

		// Set the dimension
		m_gradingDimension.setWidth(m_area.getWidth() - 4);
		m_gradingDimension.setHeight(totalHeight);
	}
	else
	{
		// The width of drawing is less than the size of the area
		const long width = m_area.getWidth() - 4;
	
		// There is a two pixel gap
		const long offset = 2 * (numberOfSteps + 1);

		// Total width of each
		const long totalWidth = (width - offset) / numberOfSteps;

		// Set the dimension
		m_gradingDimension.setWidth(totalWidth);
		m_gradingDimension.setHeight(m_area.getHeight() - 4);

	}

	// Set the colours
	this->setLineColour(CAlphaColour::CALPHACOLOUR_BLUE, CAlphaColour::CALPHACOLOUR_WHITE);
}

//	===========================================================================
CLevelMeter::~CLevelMeter()
{
	EXPONENT_CLASS_DESTRUCTION(CLevelMeter);
}

//	===========================================================================
void CLevelMeter::setValue(const double value)
{
	CControl::setValue(value);
	if (value > 1.f)
	{
		return;
	}
	else if (value < 0.0)
	{
		return;
	}
	m_numberToDraw = CParameter::doubleToLong(0, m_numberOfSteps, value);
}

//	===========================================================================
void CLevelMeter::drawControl(CGraphics &graphics)
{
	// Should we draw?
	if (!this->drawEnabledControl(graphics))
	{
		return;
	}
	
	// The are we will draw into
	CRect theArea(2, 2, m_gradingDimension.getWidth(), m_gradingDimension.getHeight());

	// Set the colours
	CPen *pen			= graphics.getMutablePen();
	CBrush *brush		= graphics.getMutableBrush();
	CAlphaColour colour = m_lineColourMin;

	// Check our orientation
	if (m_isVertical)
	{
		// We want to draw from the bottom up
		theArea.setTop(m_area.getHeight() - 2 - m_gradingDimension.getHeight());

		for (long i = 0; i < m_numberToDraw; i++)
		{
			pen->setColour(colour);
			brush->setColour(colour);
#ifdef WIN32
			// Draw the lines
			graphics.fillRoundedRectangle(theArea, CDimension(2, 2));
			graphics.drawRoundedRectangle(theArea, CDimension(2, 2));
#else
			// Draw the lines
			graphics.fillRectangle(theArea);
			graphics.drawRectangle(theArea);
#endif
			// Update the area
			theArea.setTop(theArea.getTop() - m_gradingDimension.getHeight() - 2);

			// Store the new colour
			colour.setColour(colour.getRed() + deltaPerStep[0], colour.getGreen() + deltaPerStep[1], colour.getBlue() + deltaPerStep[2], 255);
		}
	}
	else
	{
		for (long i = 0; i < m_numberToDraw; i++)
		{
			pen->setColour(colour);
			brush->setColour(colour);

#ifdef WIN32
			// Draw the lines
			graphics.fillRoundedRectangle(theArea, CDimension(2, 2));
			graphics.drawRoundedRectangle(theArea, CDimension(2, 2));
#else
			// Draw the lines
			graphics.fillRectangle(theArea);
			graphics.drawRectangle(theArea);
#endif

			// Update the area
			theArea.setLeft(theArea.getLeft() + m_gradingDimension.getWidth() + 2);

			// Store the new colour
			colour.setColour(colour.getRed() + deltaPerStep[0], colour.getGreen() + deltaPerStep[1], colour.getBlue() + deltaPerStep[2], 255);
		}
	}
}

//	===========================================================================
void CLevelMeter::setLineColour(const CAlphaColour &lineColourMin, const CAlphaColour &lineColourMax) 
{ 
	m_lineColourMin = lineColourMin; 
	m_lineColourMax = lineColourMax;

	deltaPerStep[0] = (long)((double)(m_lineColourMax.getRed()   - m_lineColourMin.getRed())   / (double)m_numberOfSteps);
	deltaPerStep[1] = (long)((double)(m_lineColourMax.getGreen() - m_lineColourMin.getGreen()) / (double)m_numberOfSteps);
	deltaPerStep[2] = (long)((double)(m_lineColourMax.getBlue()  - m_lineColourMin.getBlue())  / (double)m_numberOfSteps);
}